## =============================================================================

# TITLE: expanded_Plots.R

# DESCRIPTION: 

# These functions create plots for the 30 V, 60 V, and 90 V spectra in 
# Reporting view. For 30 V, create_ggplot_annotate is used to give annotated
# plots, while create_ggplot is used for 60 V and 90 V spectra to create
# plots without annotations. 

# TABLE OF cONTENTS: 

# 1) CREATE GGPLOTS
#     1A) 30 V GGPLOT ANNOTATED
#         - i) Create self-repelling annotations for each target
#         - ii) Create ggplot q with target labels
#     1B) 60 V/90 V GGPLOTS


## =============================================================================
## 1) CREATE GGPLOTS
## =============================================================================
## -----------------------------------------------------------------------------
## 1A) 30 V GGPLOT ANNOTATED
## -----------------------------------------------------------------------------

create_ggplot_annotate <- function(data, tit, targets) {
  
  # i) Create self-repelling annotations for each target _______________________
  # Initialize dataframe for target coordinates, label, and index
  n = length(targets)
  annotations = data.frame(
    x = numeric(n),
    y = numeric(n),
    label = character(n),
    idx = numeric(n)
  )
  
  # For each target, create label for annotations formatted "T1, T2, etc..."
  for (i in 1:n) {
    # Find index j for data
    j = which.min(abs(data[, mz] - targets[i]))
    targetlabel = ""
    targetlabel = paste0("T", as.character(i))
    # Update annotation x,y coordinates, label and index
    annotations$x[i] <- targets[i]
    annotations$y[i] <- as.numeric(unlist(data[j, 2])) + 5
    annotations$label[i] <- targetlabel
    annotations$idx[i] <- j
  }
  
  # target_annotations_labels <<- annotations$label
  rv$annotations = annotations$label
  rv$x = annotations$x
  
  # Add label field to data parameter to be used with ggrepel
  data$labels <- ""
  data$labels[annotations[, 4]] <- annotations[, 3]
  options(ggrepel.max.overlaps = Inf)
  
  
  # ii) Create ggplot q with target labels _____________________________________
  q <-
    ggplot(data, 
           aes(x = mz,xend = mz,y = ab,yend = 0,label = labels)) + 
    labs(title = tit, x = "*m/z*", y = "Relative Intensity (%)") + 
    theme(
      axis.title.x = ggtext::element_markdown(size = 14),
      axis.title.y = ggtext::element_markdown(size = 14),
      plot.title = element_text(size = 20),
      #, hjust=0.5), # To center title
      panel.grid.major = element_blank(),
      panel.grid.minor = element_blank(),
      panel.background = element_blank(),
      axis.line = element_line(color = "black")) + 
    geom_segment() + 
    geom_text_repel(size = 6) + 
    scale_x_continuous(limits = c(rv$xmin_dst, rv$xmax_dst), 
                       breaks= seq(rv$xmintick_dst, rv$xmaxtick_dst, rv$increment_dst),
                       expand = c(0,0)) +
    scale_y_continuous(expand = c(0,0)) + 
    expand_limits(x = 0, y = 0)
  
  # Add threshold line if on mixture analysis if 30 V selected
  if (rv$QuerySelected == "Low (30 V)") {
    q <- q + 
      geom_hline(size = 0.5, colour = "black",
                 aes(yintercept = 99.9 * target_min_ab_print,
                     linetype = "Threshold")) + 
      theme(legend.title = element_text(size=14, face="bold"),  
            text = element_text(size=16)) + 
      scale_linetype_manual(name = "Legend", 
                            values = c(Threshold = "dotted"))
  }
  
  # Display plot 
  return(q)
}


## -----------------------------------------------------------------------------
## 1B) 60 V/90 V GGPLOTS
## -----------------------------------------------------------------------------

# This function creates expanded view plots but without annotations
# For 60 and 90V
create_ggplot <- function(data, tit) {
  
  ggplot(data, aes(x = mz,xend = mz,y = 0,yend = ab)) +
    labs(title = tit, x = "*m/z*", y = "Relative Intensity (%)") +
    theme(
      axis.title.x = ggtext::element_markdown(size = 14),
      axis.title.y = ggtext::element_markdown(size = 14),
      plot.title = element_text(size = 20),
      #, hjust=0.5), # To center title
      panel.grid.major = element_blank(),
      panel.grid.minor = element_blank(),
      panel.background = element_blank(),
      axis.line = element_line(color = "black")) +
    geom_segment() + 
    scale_x_continuous(limits = c(rv$xmin_dst, rv$xmax_dst), 
                       breaks= seq(rv$xmintick_dst, 
                                   rv$xmaxtick_dst,
                                   rv$increment_dst),
                       expand = c(0,0)) + 
    scale_y_continuous(expand = c(0,0)) + 
    expand_limits(x = 0, y = 0)
}



