
## =============================================================================
# 
# TITLE: Trigger_Targets.R
# 
# DESCRIPTION: 
# 
# Each target is comprised of an htmlOutput and a dataTableOutput. 
# The purpose of this program is to handle everything related to inputs/outputs 
# of targets and their potential matches in interactive & reporting views, 
# outlined in 3 major steps below. 
# 
# 
# WARNING - these programs are used only to compute targets and their potential
#     matches. To see where Reporting targets are actually displayed, 
#     see mainPanel_Toggle_DST.R.
#     To see where Interactive targets are displayed, see (1Ai) and (1Aii). 
# 
# 
#                             TABLE OF cONTENTS: 
#
#     1) INPUTS / OUTPUTS: 
#         1A) INTERACTIVE TARGET OUTPUTS
#               - i) Get output for interactive targets
#               - ii) Get input for interactive targets
#               - iii) RenderUI for interactive targets
#         1B) EXPANDED TARGET OUTPUTS
#               - i) TARGET RENDERUI
#
#     2) CALCULATIONS: 
#         2A) INTERACTIVE TARGET COMPUTATION
#         2B) EXPANDED TARGET COMPUTATION
#
#     3) HELPER FUNCTIONS: 
#         3A) get_showidx - Find index for targets with potential matches
#         3B) getSigFigs - Find sigfigs for interactive/reporting
#         3C) updateTargetSigFigs - Find sigfigs for current Target
#         3D) potentialMatch_HTML - get HTMLOutput
#         3E) get_columnNames - for data table
#         OUTSIDE HELPER USED - source/ServerFunctions/getHashID.R
# 

## =============================================================================
##                          1) INPUTS / OUTPUTS: 
## =============================================================================
## -----------------------------------------------------------------------------
##                    1A) INTERACTIVE TARGET INPUTS/OUTPUTS
## -----------------------------------------------------------------------------

# i) Get output for interactive targets ________________________________________

output$interactiveTargets <- renderUI({
  # Only proceed if trigger button pressed
  if (!getTriggerButton()){return()}
  
  return(htmlOutput(outputId = "interactiveTargets"))
})

# ii) Get input for interactive targets ________________________________________

# Check currently selected tab and update reactive variable based on selection
observeEvent(input$tab, {
  # Only proceed if trigger button pressed
  if (!getTriggerButton()){return()}
  
  l = input$tab
  rv$tabSelected <- l
})

# iii) RenderUI for interactive targets ________________________________________

showtargets_UI <- function(showidx, r){
  # Compute the target in interactive view  to dispaly 
  output$targetsUI <- renderUI({
    # r shows how many targets available, only proceed if not 0
    if (r > 0){
      # Check each tab in interactive view
      do.call(tabsetPanel, c(id='tab',lapply(1:r, function(ii) {
        if (is.null(showidx)){
          # If no potential matches (probably unreachable code)
          i = ii
        } else {
          # If target, find current target potential matches
          i = showidx[ii]
        }
        tabPanel(title=paste0('Target ', i))
      })))
    } 
  })
}

## -----------------------------------------------------------------------------
##                      1B) EXPANDED TARGET OUTPUTS
## -----------------------------------------------------------------------------

# i) Get output for expanded targets ___________________________________________

# This output generates two outputs for each target (rv$numTargets), an
# HTML output and a data table output. Only the output identities are created 
# here, the outputs are computed and defined in steps (2) and (3).

output$expandedTargets <- renderUI({
  # Only proceed if trigger button pressed
  if (!getTriggerButton()){ return() }
  
  # Create an empty list
  output_list <- list()
  # Each target has two outputs. So loop through twice the number of targets
  for (i in seq(1, rv$numTargets*2, 2)){
    r_print <<- rv$numTargets
    # define id for htmlOutput
    id <- paste0("dt", i)
    # define id for dataTableOutput
    idplus <- paste0("dt", i+1)
    # Save output into output_list at index i
    output_list[[i]] = htmlOutput(outputId = id)
    output_list[[i+1]] = DT::dataTableOutput(outputId = idplus)
  }
  
  # States for managing where the app is at
  rv$state1 = TRUE
  rv$state2 = TRUE
  
  # # Hide all targets to clear the main panel from previous searches
  showHide_showidx(showidx, 1)
  
  return(output_list)
})

################################################################################
## =============================================================================
##                            2) CALCULATIONS
## =============================================================================
## -----------------------------------------------------------------------------
##                    2A) INTERACTIVE TARGET COMPUTATION
## -----------------------------------------------------------------------------

# updates interactive view HTML output and data table for each target 
showtargets_int <- function(allTargets, search_mz_tol, showidx){
  observe({
    # Create format string based on searched_mz_tol value 
    sigfigs = getSigFigs(search_mz_tol)
    sigfigfmt = paste0("%.", sigfigs, "f")
    # Get target for current tab selection using index i
    currtab = rv$tabSelected
    i = as.numeric(str_extract(currtab, "[0-9]+"))

    if (is.null(showidx)[1]){
      # No potential matches for this target
      output$interactiveTargets <- renderUI({
        HTML(potentialMatch_HTML(sigfigfmt, target_mz, i, target_in, 3))
      })
      # Get data table output when no potential matches
      output$interactivetable <- DT::renderDataTable(NULL)
      
    } else if ((allTargets[i] == "NULL")[1] || (allTargets[i][[1]] == "NA")[1]){
 
        output$interactiveTargets <- renderUI({
          HTML(potentialMatch_HTML(sigfigfmt, target_mz, i, target_in, 1))
        })
      
      # Get data table output when no potential matches
      output$interactivetable <- DT::renderDataTable(NULL)
      
    } else {
      # Get current target into data frame
      currentTarget = do.call(rbind.data.frame, allTargets[i])
      currentTarget = currentTarget[, c(1, 3, 4, 5, 6, 7)]
      
      
      
      
      # If potential matches, define columns for datatable (must go here)
      colnames(currentTarget) <- get_columnNames(target_type_print, 'interactive')
      # Define HTML output
      output$interactiveTargets <- renderUI({
        HTML(potentialMatch_HTML(sigfigfmt, target_mz, i, target_in, 2))
      })
      
      ### WARNING - sigfigs can't be adjusted because all data are strings 
      # Format data table before outputting
      currentTarget = updateTargetSigFigs(currentTarget, sigfigs, 'interactive')
      # Define dataTable ouptut
      output$interactivetable <- DT::renderDataTable({currentTarget},
                       escape = FALSE,
                       options = list(
                         dom = 'ft',
                        columnDefs = list(
                          list(width = '200px', targets = c(1, 6)),
                          # list(width = '120px', targets = c(6)),
                          list(width = '75px', targets = c(2,3,4,5))
                        ),
                        pageLength=-1
                        )
                       )
    }
  })
}

## -----------------------------------------------------------------------------
##                      2B) EXPANDED TARGET COMPUTATION
## -----------------------------------------------------------------------------

# Updates reporting view with HTML output and datatable output for each target
showtargets_exp <- function(showidx, allTargets){
  req(rv$state1 == TRUE)
  
  if (length(showidx) == 0){
    # If no targets exist, set HTML output (mode 3)
    output[["dt1"]] <- renderUI({
      HTML(potentialMatch_HTML(sigfigfmt, target_mz, i, target_in, 3))
    })
    # Get data table output when no potential matches
    output[["dt2"]] <- DT::renderDataTable(NULL)
  } else {
    
    
    # For each target in showidx
    for (i in 1:length(showidx)) {
      # Must be local to loop through i correctly in reactive context
      local({
        i <- i
        
        # Create format string based on searched_mz_tol value 
        sigfigs = getSigFigs(mz_tol_print)
        sigfigfmt = paste0("%.", sigfigs, "f")
        
        
        # Find the hashed index (since there are 2 outputs for each target)
        idx = get_hashedID(showidx, i)
        

        id = paste0("dt", idx)
        idplus = paste0("dt", idx + 1)
        
      
        
        if (allTargets[showidx[i]] == "NULL" ||
            allTargets[showidx[i]][[1]] == "NA"){

          # If target exists but with no potential matches, set HTML output
          output[[id]] <- renderUI({
            HTML(
              paste0("<br><br><br><b>Target ", showidx[i], "</b><br><br>",
                     potentialMatch_HTML(sigfigfmt,target_mz, showidx[i], target_in, 1)))
          })
          
          # Get data table output when no potential matches
          output[[idplus]] <- DT::renderDataTable(NULL)
        } else {
          
          # Get the list of targets and put into data frame
          currentTarget = do.call(rbind.data.frame, allTargets[showidx[i]])
          
          # If potential matches, define columns for datatable (must go here)
          colnames(currentTarget) <- get_columnNames(target_type_print, 'reporting')
          
          # Create the HTML output  
          output[[id]] <- renderUI({
            HTML(paste0("<br><br><br><b>Target ", showidx[i], "</b><br><br>",
                        potentialMatch_HTML(sigfigfmt, target_mz, showidx[i], target_in, 2)))
          })
          # Format data table before output
          currentTarget = updateTargetSigFigs(currentTarget, sigfigs, 'reporting')
          # Create the data table output
          output[[idplus]] <- DT::renderDataTable({currentTarget},
                                                  options = list(dom  = 't',
                                                                 columnDefs = list(
                                                                   list(width = '200px', targets = c(1, 7)),
                                                                   list(width = '120px', targets = c(2)),
                                                                   list(width = '75px', targets = c(3,4,5,6))
                                                                 )
                                                                 ), 
                                                  escape = FALSE)
        }
      })
    }
  }
}


################################################################################
## =============================================================================
##                          3) HELPER FUNCTIONS
## =============================================================================
## -----------------------------------------------------------------------------
##       3A) get_showidx - Find index for targets with potential matches
## -----------------------------------------------------------------------------

# This helper function computes an index of which targets to display. 
# If "Only show targets with potential matches" is selected, aka if hide == TRUE
# only targets with matches are shown. 
get_showidx <- function(hide, r, showidx){
  # Search through rv$numTargets to see which ones don't have matches
  if (hide == TRUE){
    hideTargetsPrint <<- TRUE
    for (i in 1:r){
      # If there is a target with matches, save this index
      if (is.null(allTargets)){
        showidx <- c()
      } else if (length(allTargets[i][[1]]) >= 1){
        showidx <- append(showidx, i)      
      }
    }
  } else {
    hideTargetsPrint <<- FALSE
    for (i in 1:r){
      # If there is a target, save this index
      showidx <- append(showidx, i)
    }
  }
  
  rv$state1 = TRUE
  return(showidx)
}


## -----------------------------------------------------------------------------
##          3B) getSigFigs - Find sigfigs for interactive/reporting
## -----------------------------------------------------------------------------

getSigFigs <- function(mz_tol_print){
  # If integer resolution
  if (mz_tol_print == 0.1){
    sigfigs = 1
  } else{
    sigfigs = 4
  }
  return(sigfigs)
}

## -----------------------------------------------------------------------------
##          3C) updateTargetSigFigs - Find sigfigs for current Target
## -----------------------------------------------------------------------------

updateTargetSigFigs <- function(currentTarget, sigfigs, mode){
 
  sigfigs3 = 3
  sigfigs4 = 4
  
  if (mode == 'interactive'){
    
    # Set columns 3, 4, 5 to 3 for result output
    currentTarget[,3]= format(round(as.numeric(currentTarget[,3]), sigfigs), 
                              nsmall=sigfigs3)
    currentTarget[,4] = format(round(as.numeric(currentTarget[,4]), sigfigs), 
                               nsmall=sigfigs3)
    currentTarget[,5] = format(round(as.numeric(currentTarget[,5]), sigfigs), 
                               scientific = FALSE,
                               nsmall=sigfigs3)
    
    currentTarget[,2]= format(round(as.numeric(currentTarget[,2]), sigfigs), 
                              scientific = FALSE,
                              nsmall=sigfigs4)
  } else if (mode == 'reporting'){
    # Set columns 3, 4, 5 to 3 for result output
    currentTarget[,4]= format(round(as.numeric(currentTarget[,4]), sigfigs), 
                              nsmall=sigfigs3)
    currentTarget[,5] = format(round(as.numeric(currentTarget[,5]), sigfigs), 
                               nsmall=sigfigs3)
    currentTarget[,6] = format(round(as.numeric(currentTarget[,6]), sigfigs), 
                               scientific = FALSE,
                               nsmall=sigfigs3)
    
    currentTarget[,3]= format(round(as.numeric(currentTarget[,3]), sigfigs), 
                              scientific = FALSE,
                              nsmall=sigfigs4)
    
    
  }
  
  return(currentTarget)
}

## -----------------------------------------------------------------------------
##                3D) potentialMatch_HTML - get HTMLOutput
## -----------------------------------------------------------------------------

# This helper function gets the HTML output for no matches (mode 1) and 
# potential matches (mode 2)
potentialMatch_HTML <- function(sigfigfmt, target_mz, i, target_in, mode){
  if (mode == 1){
    # Create HTML output with no matches in database
    paste0("Measured <em>m/z</em>: ", sprintf(sigfigfmt, target_mz(i)), "<br>",
           "Relative Intensity: ", sprintf("%0.1f",target_in(i))," %<br>",
           "No matches in database")
  } else if (mode == 2) {
    # Create HTML output with potential matches in database
    paste0("Measured <em>m/z</em>: ", sprintf(sigfigfmt, target_mz(i)), "<br>",
           "Relative Intensity: ", sprintf("%0.1f",target_in(i)), " %<br><br>",
           "Potential matches: ")
  } else if (mode == 3){
    paste0("<br>No targets with potential matches to display.<br>Please review
           the <b>Target Display</b> under settings.")
  }
}

## -----------------------------------------------------------------------------
##                3E) get_columnNames - for data table
## -----------------------------------------------------------------------------

get_columnNames <- function(target_type, mode){
  # if(target_type =="Protonated Molecule" ||
  #    target_type == "Deprotonated Molecule"){
  #   tempNames = c("Compound", "FPIE", "RevMF", mAndHPrint, 
  #                 "&#916 <em>m/z</em>", "Comments")
  # } else {
  #   # Base peak
  #   tempNames = c("Compound", "FPIE", "RevMF", "30 V Base Peak (<em>m/z</em>)", 
  #                 "&#916 <em>m/z</em>", "Comments")
  # }
  
  if (mode == 'interactive'){
    tempNames =  c("Compound", 
                   "&#916 <em>m/z</em>",
                   "FPIE", 
                   "RevMF", 
                   "IRD", 
                   "Match Type")
  } else {
    tempNames =  c("Compound", 
                   "Class",
                   "&#916 <em>m/z</em>",
                   "FPIE", 
                   "RevMF", 
                   "IRD", 
                   "Match Type")
  }
  
 
  
  return(tempNames)
}