# This program returns several score values based on the submitted reference
# mass spectrum (ref), query mass spectrum (query), mass calibration error
# (tau_ma) and the reference mass spectrums protonated molecule m/z value
# (prot_mol). tau_ma is an advanced parameter, with default value 2*0.005 m/z.
# The returned outputs are the fraction of peak intensity in the library spectrum
# explained by matching peaks in the query (PE_perc), the spread of
#  m/z value differences between reference peaks and the matching query
# peak (MD_spread), the flag which indicates whether the protonated molecule of
# the reference compound is found with any intensity in the query spectrum
# (flag), the fraction of number of peaks in the reference spectrum with matches
# in the query (nMatchedPeaks/n), the total number of peaks in the reference
# spectrum with a match in the query (nMatchedPeaks), and the total number of
# peaks in the reference spectrum (n). 
asm_ILSA_Scoring <- function(ref,query,tau_ma){
  
    x = ref
    y = query
  
    y_mz = as.numeric(y[,1][[1]])
    y_ab = as.numeric(y[,2][[1]])
    y_ab = y_ab/max(y_ab) # normalize query
    
    x_mz = as.numeric(x[,1][[1]])
    x_ab = as.numeric(x[,2][[1]])
    x_ab = x_ab/max(x_ab) # normalize reference
    
    # Gather matches
    n = length(x_ab)
    vector_for_MF_ref = numeric(n)
    vector_for_MF_que = numeric(n)
    PeaksExplained = 0;
    totAbRef = sum(x_ab)
    mass_diff = NULL;
 
    for(i in 1:n){
      s1 = abs(y_mz-x_mz[i])
      s2 = which(s1 <= tau_ma)
      
      if (length(s2)==0){
        vector_for_MF_ref[i] = x_ab[i];
        vector_for_MF_que[i] = 0;
      } else {
        vector_for_MF_ref[i] = x_ab[i];
        
        b = which.min(s1[s2])
        vector_for_MF_que[i] = y_ab[s2[b]];
        
        PeaksExplained = PeaksExplained + x_ab[i];
        mass_diff = c(mass_diff,min(s1));
        
      }
    }
  

  # Compute Reverse Match Factor
  num = sum(as.numeric(vector_for_MF_ref)*as.numeric(vector_for_MF_que))
  den = sqrt(sum(as.numeric(vector_for_MF_ref)^2)) * sqrt(sum(as.numeric(vector_for_MF_que)^2))
 
   if(den!=0){
    revMF = num/den  
  } else {
    revMF = 0
  }
  
  # Compute FPIE
  FPIE = PeaksExplained/totAbRef;
  
  # Compute MD spread (if there are more than two matched peaks)
  if(length(mass_diff) > 1){
    MD_spread = max(mass_diff)-min(mass_diff)
  } else {
    MD_spread = NA
  }
  
  results = c(FPIE, revMF, MD_spread)
  
  return(results)
  
}
