# This program returns several score values based on the submitted reference
# mass spectrum (ref), query mass spectrum (query), mass calibration error
# (tau_ma) and the reference mass spectrums protonated molecule m/z value
# (prot_mol). tau_ma is an advanced parameter, with default value 0.005 m/z.
# The returned outputs are the fraction of peak intensity in the library spectrum
# explained by matching peaks in the query (PE_perc), the standard deviation
# between m/z value difference between reference peaks and the matching query
# peak (MD_bias), the flag which indicates whether the protonated molecule of
# the reference compound is found with any intensity in the query spectrum
# (flag), the fraction of number of peaks in the reference spectrum with matches
# in the query (nMatchedPeaks/n), the total number of peaks in the reference
# spectrum with a match in the query (nMatchedPeaks), and the total number of
# peaks in the reference spectrum (n). 

asm_PeaksExplained <- function(ref,query,tau_ma,prot_mol,noise){
  x = ref
  y = query
  
  ystar  = y[which(y[,2][[1]]/max(y[,2][[1]])>noise),]
  y = ystar;
  
  n = dim(x)[1]
  totAbRef = sum(as.numeric(x[,2][[1]]));
  
  PeaksExplained = 0;
  mass_diff_biases = NULL;
  nMatchedPeaks = 0;
  
  for(i in 1:n){
    s1 = abs(as.numeric(y[,1][[1]])-as.numeric(x[i,1]))
    s2 = which(s1 <= tau_ma)
    if (length(s2)!=0){
      PeaksExplained = PeaksExplained + as.numeric(x[i,2]);
      mass_diff_biases = c(mass_diff_biases,min(s1));
      nMatchedPeaks = nMatchedPeaks+1;
    }
  }
  
  PE_perc = PeaksExplained/totAbRef;
  if(nMatchedPeaks > 1){
    MD_bias = sd(as.numeric(mass_diff_biases))
  } else if (nMatchedPeaks==1){
    MD_bias = as.numeric(mass_diff_biases)
  } else {
    MD_bias = 0.99
  }
  
  
  # check if protonated molecule is in query spectrum
  flag = 0;
  if(min(abs(as.numeric(y[,1][[1]])-prot_mol)) <= tau_ma){
    flag = 1; # protonated molecule is in query spectrum
  }
  
  results = c(PE_perc,MD_bias,flag,(nMatchedPeaks/n),nMatchedPeaks,n)
  
  return(results)
}
